## ----setup--------------------------------------------------------------------
library(ggplot2)
library(gglycan)
library(ggtangle)
library(ggstar)
library(ggrepel)

# A complex N-glycan string
s <- "Neu5Ac(a2-3)Gal(b1-4)GlcNAc(b1-2)Man(a1-3)[Neu5Ac(a2-3)Gal(b1-4)GlcNAc(b1-2)Man(a1-6)]Man(b1-4)GlcNAc(b1-4)GlcNAc"


## ----basic--------------------------------------------------------------------
gglycan(s) + 
  geom_glycan()


## ----base---------------------------------------------------------------------
p <- gglycan(s)
# p + geom_blank() # Empty canvas


## ----edges--------------------------------------------------------------------
p + geom_edge(color = "steelblue", linetype = "dashed", linewidth = 0.8)


## ----edge-labels--------------------------------------------------------------
p + 
  geom_edge(color = "grey80") +
  geom_edge_text(aes(label = label), color = "darkred", size = 3, vjust = -0.2)


## ----nodes--------------------------------------------------------------------
p + 
  geom_edge(color = "grey80") +
  geom_star(aes(x = x, y = y, starshape = I(snfg_starshape), fill = I(snfg_fill)), size = 5)


## ----text-labels--------------------------------------------------------------
p + 
  geom_edge(color = "grey80") +
  geom_star(aes(x = x, y = y, starshape = I(snfg_starshape), fill = I(snfg_fill)), size = 4) +
  geom_text_repel(aes(label = label), size = 2.5, box.padding = 0.3)


## ----customized---------------------------------------------------------------
gglycan(s, direction = "right", length = 1.5) +
  # Custom Edges
  geom_edge(color = "grey70", linewidth = 1) +
  
  # Custom Linkage Labels
  geom_edge_text(aes(label = label), color = "steelblue", size = 3.5, fontface = "bold") +
  
  # Standard SNFG Nodes
  geom_star(aes(x = x, y = y, starshape = I(snfg_starshape), fill = I(snfg_fill)), size = 5) +
  
  # Node Text Labels
  geom_text_repel(aes(label = label), size = 3, color = "black", bg.color = "white", bg.r = 0.15) +
  
  # Theme
  theme_void() +
  ggtitle("Customized Glycan Plot")


## ----motif--------------------------------------------------------------------
motif <- "Neu5Ac(a2-3)Gal(b1-4)GlcNAc"

# By default, gglycan() handles the alpha mapping if you use geom_glycan()
# But manually, we map alpha:
gglycan(s, motif = motif) +
  geom_edge(aes(alpha = I(alpha)), linewidth = 1) +
  geom_edge_text(aes(label = label, alpha = I(alpha))) +
  geom_star(aes(x = x, y = y, starshape = I(snfg_starshape), fill = I(snfg_fill), alpha = I(alpha)), size = 5)

