/*
 * Decompiled with CFR 0.152.
 */
package org.apache.james.mailbox.cassandra.mail.task;

import com.google.common.base.MoreObjects;
import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableList;
import java.util.Collection;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.ConcurrentLinkedDeque;
import java.util.concurrent.atomic.AtomicLong;
import javax.inject.Inject;
import org.apache.james.backends.cassandra.init.configuration.JamesExecutionProfiles;
import org.apache.james.backends.cassandra.versions.CassandraSchemaVersionManager;
import org.apache.james.backends.cassandra.versions.SchemaVersion;
import org.apache.james.mailbox.cassandra.ids.CassandraId;
import org.apache.james.mailbox.cassandra.mail.CassandraMailboxDAO;
import org.apache.james.mailbox.cassandra.mail.CassandraMailboxPathV3DAO;
import org.apache.james.mailbox.cassandra.mail.task.ConflictingEntry;
import org.apache.james.mailbox.model.Mailbox;
import org.apache.james.mailbox.model.MailboxId;
import org.apache.james.task.Task;
import org.reactivestreams.Publisher;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import reactor.core.publisher.Flux;
import reactor.core.publisher.Mono;

public class SolveMailboxInconsistenciesService {
    public static final Logger LOGGER = LoggerFactory.getLogger(SolveMailboxInconsistenciesService.class);
    private static final Inconsistency NO_INCONSISTENCY = (context, mailboxDAO1, pathV2DAO) -> Mono.just((Object)Task.Result.COMPLETED);
    private static final SchemaVersion MAILBOX_PATH_V_3_MIGRATION_PERFORMED_VERSION = new SchemaVersion(8);
    private final CassandraMailboxDAO mailboxDAO;
    private final CassandraMailboxPathV3DAO mailboxPathV3DAO;
    private final CassandraSchemaVersionManager versionManager;

    @Inject
    SolveMailboxInconsistenciesService(CassandraMailboxDAO mailboxDAO, CassandraMailboxPathV3DAO mailboxPathV3DAO, CassandraSchemaVersionManager versionManager) {
        this.mailboxDAO = mailboxDAO;
        this.mailboxPathV3DAO = mailboxPathV3DAO;
        this.versionManager = versionManager;
    }

    public Mono<Task.Result> fixMailboxInconsistencies(Context context) {
        this.assertValidVersion();
        return Flux.concat((Publisher[])new Publisher[]{this.processMailboxDaoInconsistencies(context), this.processMailboxPathDaoInconsistencies(context)}).reduce((Object)Task.Result.COMPLETED, Task::combine);
    }

    private void assertValidVersion() {
        SchemaVersion version = (SchemaVersion)this.versionManager.computeVersion().block();
        boolean isVersionValid = version.isAfterOrEquals(MAILBOX_PATH_V_3_MIGRATION_PERFORMED_VERSION);
        Preconditions.checkState((boolean)isVersionValid, (String)"Schema version %s is required in order to ensure mailboxPathV3DAO to be correctly populated, got %s", (int)MAILBOX_PATH_V_3_MIGRATION_PERFORMED_VERSION.getValue(), (int)version.getValue());
    }

    private Flux<Task.Result> processMailboxPathDaoInconsistencies(Context context) {
        return this.mailboxPathV3DAO.listAll().flatMap(this::detectMailboxPathDaoInconsistency, 16).flatMap(inconsistency -> inconsistency.fix(context, this.mailboxDAO, this.mailboxPathV3DAO), 16).doOnNext(any -> context.incrementProcessedMailboxPathEntries());
    }

    private Flux<Task.Result> processMailboxDaoInconsistencies(Context context) {
        return this.mailboxDAO.retrieveAllMailboxes().flatMap(this::detectMailboxDaoInconsistency, 16).flatMap(inconsistency -> inconsistency.fix(context, this.mailboxDAO, this.mailboxPathV3DAO), 16).doOnNext(any -> context.incrementProcessedMailboxEntries());
    }

    private Mono<Inconsistency> detectMailboxDaoInconsistency(Mailbox mailboxEntry) {
        Mono<Mailbox> pathEntry = this.mailboxPathV3DAO.retrieve(mailboxEntry.generateAssociatedPath(), JamesExecutionProfiles.ConsistencyChoice.STRONG);
        return Inconsistency.detectMailboxDaoInconsistency(mailboxEntry, pathEntry);
    }

    private Mono<Inconsistency> detectMailboxPathDaoInconsistency(Mailbox mailboxByPathEntry) {
        CassandraId cassandraId = (CassandraId)mailboxByPathEntry.getMailboxId();
        Mono<Mailbox> mailboxEntry = this.mailboxDAO.retrieveMailbox(cassandraId);
        return Inconsistency.detectMailboxPathDaoInconsistency(mailboxByPathEntry, mailboxEntry);
    }

    public static class Context {
        private final AtomicLong processedMailboxEntries;
        private final AtomicLong processedMailboxPathEntries;
        private final ConcurrentLinkedDeque<MailboxId> fixedInconsistencies;
        private final ConcurrentLinkedDeque<ConflictingEntry> conflictingEntries;
        private final AtomicLong errors;

        public static Builder builder() {
            return new Builder();
        }

        public Context() {
            this(new AtomicLong(), new AtomicLong(), (Collection<MailboxId>)ImmutableList.of(), (Collection<ConflictingEntry>)ImmutableList.of(), new AtomicLong());
        }

        Context(long processedMailboxEntries, long processedMailboxPathEntries, Collection<MailboxId> fixedInconsistencies, Collection<ConflictingEntry> conflictingEntries, long errors) {
            this(new AtomicLong(processedMailboxEntries), new AtomicLong(processedMailboxPathEntries), fixedInconsistencies, conflictingEntries, new AtomicLong(errors));
        }

        private Context(AtomicLong processedMailboxEntries, AtomicLong processedMailboxPathEntries, Collection<MailboxId> fixedInconsistencies, Collection<ConflictingEntry> conflictingEntries, AtomicLong errors) {
            this.processedMailboxEntries = processedMailboxEntries;
            this.processedMailboxPathEntries = processedMailboxPathEntries;
            this.fixedInconsistencies = new ConcurrentLinkedDeque<MailboxId>(fixedInconsistencies);
            this.conflictingEntries = new ConcurrentLinkedDeque<ConflictingEntry>(conflictingEntries);
            this.errors = errors;
        }

        void incrementProcessedMailboxEntries() {
            this.processedMailboxEntries.incrementAndGet();
        }

        void incrementProcessedMailboxPathEntries() {
            this.processedMailboxPathEntries.incrementAndGet();
        }

        void addFixedInconsistency(MailboxId mailboxId) {
            this.fixedInconsistencies.add(mailboxId);
        }

        void addConflictingEntries(ConflictingEntry conflictingEntry) {
            this.conflictingEntries.add(conflictingEntry);
        }

        void incrementErrors() {
            this.errors.incrementAndGet();
        }

        Snapshot snapshot() {
            return new Snapshot(this.processedMailboxEntries.get(), this.processedMailboxPathEntries.get(), (ImmutableList<MailboxId>)ImmutableList.copyOf(this.fixedInconsistencies), (ImmutableList<ConflictingEntry>)ImmutableList.copyOf(this.conflictingEntries), this.errors.get());
        }

        static class Snapshot {
            private final long processedMailboxEntries;
            private final long processedMailboxPathEntries;
            private final ImmutableList<MailboxId> fixedInconsistencies;
            private final ImmutableList<ConflictingEntry> conflictingEntries;
            private final long errors;

            private Snapshot(long processedMailboxEntries, long processedMailboxPathEntries, ImmutableList<MailboxId> fixedInconsistencies, ImmutableList<ConflictingEntry> conflictingEntries, long errors) {
                this.processedMailboxEntries = processedMailboxEntries;
                this.processedMailboxPathEntries = processedMailboxPathEntries;
                this.fixedInconsistencies = fixedInconsistencies;
                this.conflictingEntries = conflictingEntries;
                this.errors = errors;
            }

            long getProcessedMailboxEntries() {
                return this.processedMailboxEntries;
            }

            long getProcessedMailboxPathEntries() {
                return this.processedMailboxPathEntries;
            }

            ImmutableList<MailboxId> getFixedInconsistencies() {
                return this.fixedInconsistencies;
            }

            ImmutableList<ConflictingEntry> getConflictingEntries() {
                return this.conflictingEntries;
            }

            long getErrors() {
                return this.errors;
            }

            public final boolean equals(Object o) {
                if (o instanceof Snapshot) {
                    Snapshot that = (Snapshot)o;
                    return Objects.equals(this.processedMailboxEntries, that.processedMailboxEntries) && Objects.equals(this.processedMailboxPathEntries, that.processedMailboxPathEntries) && Objects.equals(this.fixedInconsistencies, that.fixedInconsistencies) && Objects.equals(this.errors, that.errors) && Objects.equals(this.conflictingEntries, that.conflictingEntries);
                }
                return false;
            }

            public final int hashCode() {
                return Objects.hash(this.processedMailboxEntries, this.processedMailboxPathEntries, this.fixedInconsistencies, this.conflictingEntries, this.errors);
            }

            public String toString() {
                return MoreObjects.toStringHelper((Object)this).add("processedMailboxEntries", this.processedMailboxEntries).add("processedMailboxPathEntries", this.processedMailboxPathEntries).add("fixedInconsistencies", this.fixedInconsistencies).add("conflictingEntries", this.conflictingEntries).add("errors", this.errors).toString();
            }
        }

        static class Builder {
            private Optional<Long> processedMailboxEntries;
            private Optional<Long> processedMailboxPathEntries = Optional.empty();
            private ImmutableList.Builder<MailboxId> fixedInconsistencies = ImmutableList.builder();
            private ImmutableList.Builder<ConflictingEntry> conflictingEntries = ImmutableList.builder();
            private Optional<Long> errors = Optional.empty();

            Builder() {
                this.processedMailboxEntries = Optional.empty();
            }

            public Builder processedMailboxEntries(long count) {
                this.processedMailboxEntries = Optional.of(count);
                return this;
            }

            public Builder processedMailboxPathEntries(long count) {
                this.processedMailboxPathEntries = Optional.of(count);
                return this;
            }

            public Builder addFixedInconsistencies(MailboxId mailboxId) {
                this.fixedInconsistencies.add((Object)mailboxId);
                return this;
            }

            public Builder addConflictingEntry(ConflictingEntry conflictingEntry) {
                this.conflictingEntries.add((Object)conflictingEntry);
                return this;
            }

            public Builder errors(long count) {
                this.errors = Optional.of(count);
                return this;
            }

            public Context build() {
                return new Context(this.processedMailboxEntries.orElse(0L), this.processedMailboxPathEntries.orElse(0L), (Collection<MailboxId>)this.fixedInconsistencies.build(), (Collection<ConflictingEntry>)this.conflictingEntries.build(), this.errors.orElse(0L));
            }
        }
    }

    private static class ConflictingEntryInconsistency
    implements Inconsistency {
        private final ConflictingEntry conflictingEntry;

        private ConflictingEntryInconsistency(ConflictingEntry conflictingEntry) {
            this.conflictingEntry = conflictingEntry;
        }

        @Override
        public Mono<Task.Result> fix(Context context, CassandraMailboxDAO mailboxDAO, CassandraMailboxPathV3DAO pathV3DAO) {
            LOGGER.error("MailboxDAO contains mailbox {} {} which conflict with corresponding registration {} {}. We recommend merging these mailboxes together to prevent mail data loss.", new Object[]{this.conflictingEntry.getMailboxDaoEntry().getMailboxId(), this.conflictingEntry.getMailboxDaoEntry().getMailboxPath(), this.conflictingEntry.getMailboxPathDaoEntry().getMailboxId(), this.conflictingEntry.getMailboxPathDaoEntry().getMailboxPath()});
            context.addConflictingEntries(this.conflictingEntry);
            return Mono.just((Object)Task.Result.PARTIAL);
        }
    }

    private static class OrphanMailboxPathDAOEntry
    implements Inconsistency {
        private final Mailbox mailbox;

        private OrphanMailboxPathDAOEntry(Mailbox mailbox) {
            this.mailbox = mailbox;
        }

        @Override
        public Mono<Task.Result> fix(Context context, CassandraMailboxDAO mailboxDAO, CassandraMailboxPathV3DAO pathV3DAO) {
            return pathV3DAO.delete(this.mailbox.generateAssociatedPath()).doOnSuccess(any -> {
                LOGGER.info("Inconsistency fixed for orphan mailboxPath {} - {}", (Object)this.mailbox.getMailboxId().serialize(), (Object)this.mailbox.generateAssociatedPath().asString());
                context.addFixedInconsistency(this.mailbox.getMailboxId());
            }).map(any -> Task.Result.COMPLETED).defaultIfEmpty((Object)Task.Result.COMPLETED).onErrorResume(e -> {
                LOGGER.error("Failed fixing inconsistency for orphan mailboxPath {} - {}", new Object[]{this.mailbox.getMailboxId().serialize(), this.mailbox.generateAssociatedPath().asString(), e});
                context.incrementErrors();
                return Mono.just((Object)Task.Result.PARTIAL);
            });
        }
    }

    private static class OrphanMailboxDAOEntry
    implements Inconsistency {
        private final Mailbox mailbox;

        private OrphanMailboxDAOEntry(Mailbox mailbox) {
            this.mailbox = mailbox;
        }

        @Override
        public Mono<Task.Result> fix(Context context, CassandraMailboxDAO mailboxDAO, CassandraMailboxPathV3DAO pathV3DAO) {
            return pathV3DAO.save(this.mailbox).map(success -> {
                if (success.booleanValue()) {
                    this.notifySuccess(context);
                    return Task.Result.COMPLETED;
                }
                this.notifyFailure(context);
                return Task.Result.PARTIAL;
            });
        }

        private void notifyFailure(Context context) {
            context.incrementErrors();
            LOGGER.warn("Failed fixing inconsistency for orphan mailbox {} - {}", (Object)this.mailbox.getMailboxId().serialize(), (Object)this.mailbox.generateAssociatedPath().asString());
        }

        private void notifySuccess(Context context) {
            LOGGER.info("Inconsistency fixed for orphan mailbox {} - {}", (Object)this.mailbox.getMailboxId().serialize(), (Object)this.mailbox.generateAssociatedPath().asString());
            context.addFixedInconsistency(this.mailbox.getMailboxId());
        }
    }

    @FunctionalInterface
    public static interface Inconsistency {
        public static Mono<Inconsistency> detectMailboxDaoInconsistency(Mailbox mailboxEntry, Mono<Mailbox> pathEntry) {
            return pathEntry.map(mailboxByPath -> {
                if (mailboxByPath.getMailboxId().equals(mailboxEntry.getMailboxId())) {
                    return NO_INCONSISTENCY;
                }
                return new ConflictingEntryInconsistency(ConflictingEntry.builder().mailboxDaoEntry(mailboxEntry).mailboxPathDaoEntry((Mailbox)mailboxByPath));
            }).defaultIfEmpty((Object)new OrphanMailboxDAOEntry(mailboxEntry));
        }

        public static Mono<Inconsistency> detectMailboxPathDaoInconsistency(Mailbox mailboxByPathEntry, Mono<Mailbox> mailboxEntry) {
            return mailboxEntry.map(mailboxById -> {
                if (mailboxByPathEntry.generateAssociatedPath().equals((Object)mailboxById.generateAssociatedPath())) {
                    return NO_INCONSISTENCY;
                }
                return new ConflictingEntryInconsistency(ConflictingEntry.builder().mailboxDaoEntry((Mailbox)mailboxById).mailboxPathDaoEntry(mailboxByPathEntry));
            }).defaultIfEmpty((Object)new OrphanMailboxPathDAOEntry(mailboxByPathEntry));
        }

        public Mono<Task.Result> fix(Context var1, CassandraMailboxDAO var2, CassandraMailboxPathV3DAO var3);
    }
}

